program GAUSSIANQUADRATURE;
{--------------------------------------------------------------------}
{  Alg7'6.pas   Pascal program for implementing Algorithm 7.6        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 7.6 (Gauss-Legendre Quadrature).                        }
{  Section   7.5, Gauss-Legendre Integration, Page 397               }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    FunMax = 9;
    Epsilon = 1E-8;

  type
    LETTER = string[1];
    LETTERS = string[200];
    VECTOR = array[1..17] of real;
    MATRIX = array[1..17, 1..96] of real;
    BIGVEC = array[1..96] of real;
    States = (Changes, Done, Working);
    DoSome = (Go, Stop);

  var
    FileVar: TEXT;
    FunType, I, Inum, J, L, Sub: integer;
    A0, B0, Close1, Rnum, Tol: real;
    A, W: MATRIX;
    Q: VECTOR;
    VF, VX: BIGVEC;
    Line, ExLine: string[255];
    FileName: string[14];
    Under: BOOLEAN;
    Ans: CHAR;
    Mess: LETTERS;
    State: States;
    DoMo: DoSome;

  function F (var X: real): real;
  begin
    case FunType of
      0:
        begin
          if X <> 0 then
            F := SIN(X) / X
          else
            F := 1;
        end;
      1: 
        begin
          if X <> 0 then
            F := 1 / X
          else
            begin
              if A0 = 0 then
                F := 1E37;
              if B0 = 0 then
                F := -1E37;
              if A0 * B0 < 0 then
                F := 0;
            end;
        end;
      2: 
        F := SIN(X);
      3: 
        F := 4 / (1 + X * X);
      4: 
        F := X / (1 + X * X);
      5: 
        begin
          if 2 * X <> 7 then
            F := 1 / (7 - 2 * X)
          else
            begin
              if A0 = 3.5 then
                F := -1E37;
              if B0 = 3.5 then
                F := 1E37;
              if (2 * A0 - 7) * (2 * B0 - 7) < 0 then
                F := 0;
            end;
        end;
      6: 
        begin
          if X = 3.5 then
            F := -1E37
          else
            F := LN(ABS(X));
        end;
      7: 
        F := X * EXP(-X);
      8: 
        F := 64 * (X - 1) * EXP(-X * LN(4));
      9: 
        F := EXP(-X * X / 2) / SQRT(2 * PI);
    end;
  end;

  procedure PRINTFUNCTION (FunType: integer);
  begin
    case FunType of
      0: 
        WRITE('SIN(X)/X');
      1:
        WRITE('1/X');
      2: 
        WRITE('SIN(X)');
      3: 
        WRITE('4/(1+X^2)');
      4: 
        WRITE('X/(1+X^2)');
      5: 
        WRITE('1/(7-2*X)');
      6: 
        WRITE('LN|X|');
      7: 
        WRITE('X*EXP(-X)');
      8: 
        WRITE('64*(X-1)*4^-X');
      9: 
        WRITE('EXP(-X^2/2)/SQRT(2*Pi)');
    end;
  end;

  procedure GAUSSQUAD ( {FUNCTION F(X:real): real;}
                  A0, B0: real; var Q: VECTOR; var Close1: real; var J: integer);
    const
      Min = 3;
    var
      I, K: integer;
      Mid, Sum, Wide, X: real;
  begin
    Mid := (A0 + B0) / 2;
    Wide := (B0 - A0) / 2;
    Close1 := 1;
    J := 1;
    X := Mid + A[1, 1];
    Q[1] := W[1, 1] * F(X) * Wide;
    while ((Close1 > Tol) or (J < Min)) and (J < 17) do
      begin
        J := J + 1;
        Sum := 0;
        I := J;
        if J > 10 then
          I := 12 + 4 * (J - 11);
        if J > 14 then
          I := 24 + 8 * (J - 14);
        for K := 1 to I do
          begin
            X := Mid + A[J, K] * Wide;
            Sum := Sum + W[J, K] * F(X);
          end;
        Q[J] := Sum * Wide;
        Close1 := ABS(Q[J] - Q[J - 1]);
      end;
  end;

  procedure INPUT (var FunType: integer);
    var
      K: integer;
      Resp: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('          Gaussian quadrature is performed to find an');
    WRITELN;
    WRITELN('     approximation for the value of the definite integral:');
    WRITELN;
    WRITELN('                   B              N           ');
    WRITELN('                   /                          ');
    WRITELN('                   | F(X) DX  ~  SUM  w F(x ) ');
    WRITELN('                   /                   k   k  ');
    WRITELN('                   A             k=1          ');
    WRITELN;
    WRITELN('     Choose your function:');
    WRITELN;
    for K := 0 to FunMax do
      begin
        WRITE('             <', K : 2, ' >   F(X) = ');
        PRINTFUNCTION(K);
        WRITELN;
      end;
    WRITELN;
    WRITE('                     Select < 0 - ', FunMax : 1, ' > ?  ');
    FunType := 0;
    READLN(FunType);
    if FunType < 0 then
      FunType := 0;
    if FunType > FunMax then
      FunType := FunMax;
  end;

  procedure MESSAGE (var Tol: real);
    var
      K: integer;
      Resp: real;
  begin
    CLRSCR;
    WRITELN('          Gaussian quadrature is performed to find an approximation for');
    WRITELN;
    WRITELN('     the value of the definite integral:');
    WRITELN;
    WRITELN('               B              N  ');
    WRITELN('               /                 ');
    WRITELN('               | f(x) dx  ~  SUM  w   f(x   )  =  G(f,N),');
    WRITELN('               /                   N,k   N,k  ');
    WRITELN('               A             k=1 ');
    WRITELN;
    WRITELN;
    WRITELN('     where the abscissas {x   :k=1,2,...,N} are translations of the zeros');
    WRITELN('                           N,k');
    WRITELN;
    WRITELN('     of the degree N Legendre polynomial on [-1,1] to the interval [a,b].');
    WRITELN;
    WRITELN;
    WRITELN('     Successive approximations G(f,N) are computed until ');
    WRITELN;
    WRITELN;
    WRITELN('                  |G(f,N)-G(f,N-1)| < TOL. ');
    WRITELN;
    WRITELN;
    Mess := '                   ENTER the value    TOL = ';
    Tol := Epsilon;
    WRITE(Mess);
    READLN(Tol);
    Tol := ABS(Tol);
    if (Tol < Epsilon) then
      Tol := Epsilon;
  end;

  procedure PROBLEM (FunType: integer);
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     You chose to approximate the definite integral:');
    WRITELN;
    WRITELN('             B');
    WRITELN('             /');
    WRITE('             | ');
    PRINTFUNCTION(FunType);
    WRITELN(' DX');
    WRITELN('             /');
    WRITELN('             A');
    WRITELN;
  end;

  procedure EPOINTS (var A, B: real; var State: STATES);
    type
      STATUS = (Change, Enter, Done);
      LETTER = string[1];
    var
      Valu: real;
      Resp: LETTER;
      Stat: STATUS;
  begin
    Stat := Enter;
    if State = Changes then
      Stat := Change;
    while (Stat = Enter) or (Stat = Change) do
      begin
        PROBLEM(FunType);
        WRITELN;
        WRITELN('     The tolerance value is TOL =', Tol : 15 : 8);
        if (Stat = Enter) then
          begin
            WRITELN;
            WRITELN('     Now you must choose the interval [A,B].');
            ;
            Mess := '             ENTER  the left  endpoint A = ';
            A := 0;
            WRITE(Mess);
            READLN(A);
            Mess := '             ENTER  the right endpoint B = ';
            B := 1;
            WRITE(Mess);
            READLN(B);
          end
        else
          begin
            WRITELN;
            WRITELN('     The  left  endpoint  is  A =', A : 15 : 7);
            WRITELN;
            WRITELN('     The  right endpoint  is  B =', B : 15 : 7);
          end;
        WRITELN;
        WRITE('     Do you want to make a change ? <Y/N> ');
        READLN(Resp);
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Change;
            PROBLEM(FunType);
            WRITELN;
            WRITELN('The  current tolerance  is  TOL =', Tol : 15 : 7);
            Mess := 'ENTER  the  NEW  value  of  TOL = ';
            WRITE(Mess);
            READLN(Tol);
            Tol := ABS(Tol);
            if (Tol < Epsilon) then
              Tol := Epsilon;
            WRITELN;
            WRITELN('The current left  endpoint is A =', A : 15 : 7);
            Mess := 'ENTER the  NEW left  endpoint A = ';
            WRITE(Mess);
            READLN(A);
            WRITELN;
            WRITELN('The current right endpoint is B =', B : 15 : 7);
            Mess := 'ENTER the  NEW right endpoint B = ';
            WRITE(Mess);
            READLN(B);
            WRITELN;
          end
        else
          Stat := Done;
      end;
  end;

  procedure MORETABLE ( {FUNCTION F(X:real): real;}
                  A0, B0: real; var Q: VECTOR; var J: integer);
    var
      I, K: integer;
      Mid, QR, Sum, Wide, X: real;
  begin
    Mid := (A0 + B0) / 2;
    Wide := (B0 - A0) / 2;
    X := Mid + A[1, 1];
    QR := W[1, 1] * F(X) * Wide;
    Sum := 0;
    I := J;
    if J > 10 then
      I := 12 + 4 * (J - 11);
    if J > 14 then
      I := 24 + 8 * (J - 14);
    for K := 1 to I do
      begin
        X := Mid + A[J, K] * Wide;
        VX[I - K + 1] := X;                 {extra output}
        ;
        VF[I - K + 1] := W[J, K] * Wide * F(X);  {extra output}
        ;
        Sum := Sum + W[J, K] * F(X);
      end;
    QR := Sum * Wide;
    CLRSCR;
    WRITELN;
    for K := 1 to I do
      begin
        WRITELN('x  = ', VX[K] : 15 : 7, '   w  = ', W[J, K] * Wide : 15 : 7, '  w  f(x  ) = ', VF[K] : 15 : 7);
        if K < 10 then
          begin
            WRITE(' ', K : 1, '                      ', K : 1);
            WRITELN('                     ', K : 1, '    ', K : 1);
          end
        else
          begin
            WRITE(' ', K : 1, '                     ', K : 1);
            WRITELN('                    ', K : 1, '   ', K : 1);
          end;
        if K mod 11 = 9 then
          begin
            WRITE('                  Press the  <ENTER>  key.  ');
            READLN(Ans);
            WRITELN;
            WRITELN
          end;
      end;
    WRITELN('The   Gaussian   quadrature   approximation  is          ', QR : 15 : 7);
    WRITELN;
    WRITE('                  Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
    WRITELN;
  end;

  procedure GTABLE (Q: VECTOR; J: integer);
    type
      GTStates = (Looking, Done);
    var
      I, K: integer;
      Resp: CHAR;
      GState: GTStates;
  begin
    GState := Looking;
    while GState = Looking do
      begin
        CLRSCR;
        WRITELN('     The table of  Gauss-Legendre  quadrature approximations is:');
        WRITELN;
        WRITELN('     Line      Number of Points    Approx. for integral');
        if J < 16 then
          WRITELN;
        WRITELN('       K              N                 Q(f,N)');
        WRITELN('     ---------------------------------------------------');
        if J < 17 then
          WRITELN;
        for K := 1 to J do
          begin
            I := K;
            if K > 10 then
              I := 12 + 4 * (K - 11);
            if K > 14 then
              I := 24 + 8 * (K - 14);
            WRITELN('     ', K : 3, '            ', I : 3, '         ', Q[K] : 15 : 7);
          end;
        WRITELN;
        WRITE('    Want to see more of the calculations ?  <Y/N>  ');
        READLN(Resp);
        WRITELN;
        WRITELN;
        WRITELN;
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Mess := '    Enter the line number  K = ';
            L := 1;
            WRITE(Mess);
            READLN(L);
            if L < 1 then
              L := 1;
            if L > J then
              L := J;
            MORETABLE(A0, B0, Q, L);
          end
        else
          begin
            WRITELN;
            WRITELN;
          end;
        if (Resp <> 'Y') and (Resp <> 'y') then
          GState := Done;
      end;
  end;

  procedure RESULTS (A0, B0, Close1: real; Q: VECTOR; J: integer);
    var
      I, II, JJ, K, L, U, V: integer;
  begin
    JJ := J;
    if J > 10 then
      JJ := 12 + 4 * (J - 11);
    if J > 14 then
      JJ := 24 + 8 * (J - 14);
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN(B0 : 13 : 5);
    WRITELN('       /');
    WRITE('       |  ');
    PRINTFUNCTION(FunType);
    WRITELN(' DX ~', Q[J] : 15 : 7);
    WRITELN('       /');
    WRITELN(A0 : 13 : 5);
    WRITELN;
    WRITELN('Gaussian quadrature with', JJ : 3, ' points was used to integrate');
    WRITELN;
    WRITE('F(X) = ');
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN;
    WRITELN('over the interval   [', A0 : 15 : 7, '  ,', B0 : 15 : 7, '  ].');
    WRITELN;
    WRITELN('Some values of the previous Gaussian rules are:');
    WRITELN;
    L := 1;
    if J > 10 then
      L := 2;
    if J > 12 then
      L := 3;
    if J > 14 then
      L := 4;
    for K := L to TRUNC((J + 1) / 2) do
      begin
        U := 2 * K - 1;
        V := 2 * K;
        I := U;
        if U > 10 then
          I := 12 + 4 * (U - 11);
        if U > 14 then
          I := 24 + 8 * (U - 14);
        II := V;
        if V > 10 then
          II := 12 + 4 * (V - 11);
        if V > 14 then
          II := 24 + 8 * (V - 14);
        if U <= J then
          begin
            WRITE('    Q(', I : 1, ') =', Q[U] : 15 : 7, '      ');
            if V <= J then
              WRITELN('Q(', II : 1, ') =', Q[V] : 15 : 7)
            else
              WRITELN;
          end;
      end;
    WRITELN;
    WRITELN('The best Gaussian approximation is Q(', JJ : 2, ') =', Q[J] : 15 : 7);
    if J <= 11 then
      WRITELN;
    WRITELN('                 The error estimate is  +-', Close1 : 15 : 7);
  end;

  procedure READAW;
    var
      I, J, K: integer;
      M: real;
      InputFile: TEXT;
      FileExists: BOOLEAN;
  begin
     {$I-}
    FileName := 'ALG7-6AW.DAT';
    ASSIGN(InputFile,FileName);
    RESET(InputFile);
     {$I+}
    FileExists := (IOResult = 0);
    WRITELN;
    WRITELN;
    WRITE('               ');
    if FileExists then
      WRITELN('The File Exists.')
    else
      WRITELN('The File Was Not Found.');
    for J := 1 to 17 do
      begin
        M := J;
        I := J;
        if (J > 10) then
          begin
            M := 12 + 4 * (J - 11);
            I := TRUNC(M);
          end;
        if (J > 14) then
          begin
            M := 24 + 8 * (J - 14);
            I := TRUNC(M);
          end;
        for K := 1 to TRUNC((M + 1) / 2) do
          READLN(InputFile, A[J, K]);
        for K := 1 to TRUNC(M / 2) do
          A[J, I + 1 - K] := -A[J, K];
        for K := 1 to TRUNC((M + 1) / 2) do
          READLN(InputFile, W[J, K]);
        for K := 1 to TRUNC(M / 2) do
          W[J, I + 1 - K] := W[J, K];
      end;
        CLOSE(InputFile);
  end;

begin                                            {Begin Main Program}
  CLRSCR;
  for I := 1 to 10 do
    WRITELN;
  WRITE('               Now loading abscissas and weights. Please wait!  ');
  READAW;
  MESSAGE(Tol);
  DoMo := Go;
  while DoMo = Go do
    begin
      INPUT(FunType);
      State := Working;
      while (State = Working) or (State = Changes) do
        begin
          EPOINTS(A0, B0, State);
          GAUSSQUAD(A0, B0, Q, Close1, J);
          RESULTS(A0, B0, Close1, Q, J);
          WRITELN;
          WRITE('Want to see the  table of approximations ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'y') or (Ans = 'Y') then
            GTABLE(Q, J);
          WRITELN;
          WRITE('Want  to try  another  interval  or  TOL ?  <Y/N>  ');
          READLN(Ans);
          WRITELN;
          if (Ans = 'Y') or (Ans = 'y') then
            State := Changes;
          if (Ans <> 'Y') and (Ans <> 'y') then
            State := Done;
        end;
      WRITELN;
      WRITE('Do you want to use  a different function ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        DoMo := Stop;
    end;                                          {End of Main Program}
end.

